#!/bin/csh -f
###################################################
#                                                 #
#  stub.csh -- Generic executable calling stub    #
#              which determines system type and   #
#              calls appropriate executable.      #
#              Make a symbolic link to this file  #
#              (or rename this file) with the     #
#              name of the executable to be       #
#              called.                            #
#                                                 #
#  Copyright (c) Altera Corporation 1995          #
#  All Rights Reserved                            #
#                                                 #
###################################################

# Use the MAX2_HOME environment variable to override
#   the installation directory.
#

# Use the MAX2_PLATFORM environment variable to
#   override the platform type.  Use only if this
#   script cannot determine it correctly.
#

# This script is generic, and simply 
#   linking the stub.csh with "ln -s" will cause
#   it to know which executable to call.  
#   If for some reason this doesn't work,
#   copy this file to the executable name, and
#   change the "set exe_name" line below.
#
set exe_name  = `basename $0`

#
# Now determine the name of the directory which 
# this file is in, and set $max2_home to be one
# level above it.
#

if (! $?MAX2_HOME) then
	set script_path = `which $0 |& tail -1`

	if ("$script_path" != "") then
		set max2_home = `dirname $script_path`/..
	else
		set max2_home = ..
	endif
else
	set max2_home = $MAX2_HOME
endif

if (! $?HOME) then
	echo "The HOME environment variable must be set in order to use $exe_name."
	exit
else if ($exe_name == maxplus2) then
	if (! $?MAXPLUS2_INI) then
		if (! -f $HOME/maxplus2.ini) then
			echo "Copying default maxplus2.ini file from $max2_home to $HOME..."
			cp $max2_home/maxplus2.ini $HOME
			chmod +w $HOME/maxplus2.ini
		endif
	else
		if (! -f $MAXPLUS2_INI/maxplus2.ini) then
			echo "Copying default maxplus2.ini file from $max2_home to $MAXPLUS2_INI..."
			cp $max2_home/maxplus2.ini $MAXPLUS2_INI >& /dev/null
			chmod +w $MAXPLUS2_INI/maxplus2.ini >& /dev/null
		endif
	endif
endif

#
# Now determine what type of system this is,
# set the $host_platform, and set the LD_LIBRARY_PATH
# variable.
#

if (! $?MAX2_PLATFORM ) then

	set os_type=`uname -s`
	set os_version=`uname -r`

	if ($os_type == "SunOS") then
		if ($os_version =~ 5.*) then
			# This is solaris 2.x
			set host_platform=solaris
		else if ($os_version =~ 4.1.*) then
			# This is sunos 4.1.x
			set host_platform=sunos
		else
			echo ""
			echo "Unable to determine the type of system you are running."
		endif
	else if ($os_type == "HP-UX") then
		# This is an HP
		set host_platform="hp"
	else if ($os_type == "AIX") then
		# This is an RS/6000
		set host_platform="rs6000"
	else
		echo ""
		echo "Unable to determine the type of system you are running."
	endif
else
	switch ($MAX2_PLATFORM)
	case solaris:
	case sunos:
	case hp:
	case rs6000:
		set host_platform = $MAX2_PLATFORM
		breaksw
	default:
		echo ""
		echo "The environment variable MAX2_PLATFORM is set to $MAX2_PLATFORM,"
		echo "but $MAX2_PLATFORM is an invalid platform type."
		breaksw
	endsw
endif

if (! $?host_platform) then
	echo ""
	echo "Use the environment variable MAX2_PLATFORM to specify"
	echo "one of the following:"
	echo ""
	echo "      sunos       (SunOS 4.1.2+ / Solaris 1.x)"
	echo "      solaris     (SunOS 5.4+   / Solaris 2.4+)"
	echo "      hp          (HP-UX 9.03+)"
	echo "      rs6000      (AIX 3.2.5+)"
	echo ""
	echo "Unable to run $exe_name."
	echo ""
	exit 1
else
	# set up the shared library path
	switch ($host_platform)
	case solaris:
	case sunos:
		if (! $?LD_LIBRARY_PATH) then
			setenv LD_LIBRARY_PATH ""
		endif
		setenv LD_LIBRARY_PATH $max2_home/$host_platform":"$LD_LIBRARY_PATH
		breaksw
	case hp:
		if (! $?SHLIB_PATH) then
			setenv SHLIB_PATH ""
		endif
		setenv SHLIB_PATH $max2_home/$host_platform":"$SHLIB_PATH
		breaksw
	case rs6000:
		if (! $?LIBPATH) then
			setenv LIBPATH ""
		endif
		setenv LIBPATH $max2_home/$host_platform":"$LIBPATH
		breaksw
	default:
		echo ""
		echo "The environment variable MAX2_PLATFORM is set to $MAX2_PLATFORM,"
		echo "but $MAX2_PLATFORM is an invalid platform type."
		breaksw
	endsw
endif

#
# Set MainWin required environment variables.
#

setenv MWHOME $max2_home/$host_platform
setenv MWWINHELP_FILE_SEARCH_PATH $max2_home/help
setenv MWWINHELP_OPEN_DIRECTORY   $max2_home/help
setenv MWDONT_USE_CDE_COLORS True

#
# Now call the real executable.
#
@ i=1
while ($i <= $#argv)
	if (`echo "$argv[$i]" | wc -w` != 1) then
		set j = ("$argv[$i]")
		set argv[$i] = \""$j"\"
	endif
	@ i = $i + 1
end

if (! -d $max2_home/$host_platform || (! -f $max2_home/$host_platform/$exe_name.exe && ! -f $max2_home/$host_platform/$exe_name)) then
	echo ""
	echo "$exe_name was not installed for $host_platform"
	echo "in the directory $max2_home"
	echo ""
	echo "If $exe_name was installed somewhere other than $max2_home, you"
	echo "can override $max2_home by using the MAX2_HOME environment variable."
	echo ""
	echo "If $exe_name is installed for a platform other than $host_platform, you"
	echo "can override $host_platform by using the MAX2_PLATFORM environment"
	echo "variable."
	echo ""
	echo "Unable to run $exe_name"
	echo ""
	exit 1
else if ($exe_name == genmem) then
	$max2_home/$host_platform/$exe_name.exe $*
else if ($exe_name =~ lm*) then
	$max2_home/$host_platform/$exe_name $*
else
	 if ($exe_name == winhelp) then
		setenv MWSCALABLE_FONTS True
		$max2_home/$host_platform/mwfontcache.exe -c >& /dev/null
	else if ($exe_name == mwfontcache) then
		$max2_home/$host_platform/$exe_name $*
	else
		setenv MWSCALABLE_FONTS False
		$max2_home/$host_platform/mwfontcache.exe -c >& /dev/null
	endif
	
	exec echo "$max2_home/$host_platform/$exe_name.exe $*" | /bin/csh -ef
endif
